-- ============================================================================
-- AutoOpenSettings.lua - Settings System for AutoOpen
-- 
-- Author: GulliDeckel
-- Version: 1.1.0.0
-- 
-- Features:
-- - XML-based settings storage
-- - Default values fallback
-- - Automatic file creation
-- - Convenience getter/setter functions
-- ============================================================================

AutoOpenSettings = {}

-- Default Settings
AutoOpenSettings.DEFAULT_SETTINGS = {
    autoOpenEnabled = true,
    playerAutoOpenEnabled = true,
    vehicleAutoOpenEnabled = true,
    closeDelayMs = 3000,
    debugEnabled = false
}

-- Settings file path
AutoOpenSettings.SETTINGS_FILE = getUserProfileAppPath() .. "modSettings/FS25_autoOpen/settings.xml"

-- Current settings (initialized with defaults)
AutoOpenSettings.settings = {}
for key, value in pairs(AutoOpenSettings.DEFAULT_SETTINGS) do
    AutoOpenSettings.settings[key] = value
end

-- Initialize settings system
function AutoOpenSettings:initialize()
    debugPrint("AutoOpenSettings initializing", "SETTINGS")
    
    -- Create directory if needed
    local directory = self.SETTINGS_FILE:match("(.*/)")
    if directory then
        createFolder(directory)
    end
    
    -- Load settings
    self:loadSettings()
    
    debugPrint("AutoOpenSettings initialized", "SETTINGS")
end

-- Load settings from XML file
function AutoOpenSettings:loadSettings()
    debugPrint("Loading settings from: " .. self.SETTINGS_FILE, "SETTINGS")
    
    if not fileExists(self.SETTINGS_FILE) then
        debugPrint("Settings file doesn't exist - using defaults", "SETTINGS")
        self:saveSettings() -- Create file with defaults
        return
    end
    
    local xmlFile = XMLFile.load("autoOpenSettings", self.SETTINGS_FILE)
    if xmlFile == nil then
        debugPrint("Could not load settings XML", "SETTINGS")
        return
    end
    
    -- Load all settings
    for key, defaultValue in pairs(self.DEFAULT_SETTINGS) do
        local xmlPath = "settings." .. key
        local value = xmlFile:getValue(xmlPath, defaultValue)
        self.settings[key] = value
        debugPrint("Setting loaded: " .. key .. " = " .. tostring(value), "SETTINGS")
    end
    
    xmlFile:delete()
    debugPrint("Settings loaded successfully", "SETTINGS")
end

-- Save settings to XML file
function AutoOpenSettings:saveSettings()
    debugPrint("Saving settings", "SETTINGS")
    
    local xmlFile = XMLFile.create("autoOpenSettings", self.SETTINGS_FILE, "settings")
    if xmlFile == nil then
        debugPrint("Could not create settings XML", "SETTINGS")
        return
    end
    
    -- Save all settings
    for key, value in pairs(self.settings) do
        local xmlPath = "settings." .. key
        xmlFile:setValue(xmlPath, value)
        debugPrint("Setting saved: " .. key .. " = " .. tostring(value), "SETTINGS")
    end
    
    xmlFile:save()
    xmlFile:delete()
    debugPrint("Settings saved successfully", "SETTINGS")
end

-- Get setting value
function AutoOpenSettings:getSetting(key)
    return self.settings[key]
end

-- Set setting value
function AutoOpenSettings:setSetting(key, value)
    if self.settings[key] ~= nil then
        self.settings[key] = value
        self:saveSettings()
        debugPrint("Setting changed: " .. key .. " = " .. tostring(value), "SETTINGS")
    end
end

-- Convenience functions
function AutoOpenSettings:isAutoOpenEnabled()
    return self.settings.autoOpenEnabled
end

function AutoOpenSettings:isPlayerAutoOpenEnabled()
    return self.settings.playerAutoOpenEnabled
end

function AutoOpenSettings:isVehicleAutoOpenEnabled()
    return self.settings.vehicleAutoOpenEnabled
end

function AutoOpenSettings:getCloseDelayMs()
    return self.settings.closeDelayMs
end

function AutoOpenSettings:isDebugEnabled()
    return self.settings.debugEnabled
end

-- Initialize settings system
AutoOpenSettings:initialize()

-- debugPrint("AutoOpenSettings System loaded", "LOADING")